module net.BurtonRadons.dedit.document;

import net.BurtonRadons.dedit.syntaxHighlighter;

class Document
{
    static int newIndex = 1;

    char [] [] lines; /* Lines of text. */
    char [] [] highs; /* Highlighting data (each char is a syntax coloring index). */
        /* Syntax coloring indices:
         *   '*' - comment.
         *   '#' - number.
         *   '"' - std.string.
         *   'i' - normal identifier.
         *   'm' - special identifier.
         *   'r' - reserved keyword.
         *   's' - symbol.
         */

    SyntaxHighlighter hilite; /* Highlighter for this document. */
    int hiliteExtra; /* Extra bytes for this highlighter's lines. */
    bit dirty = true;
    char [] filename;
    int index; /* Name index for new files. */
    bit modified = false;
    char [] endline = "\n";

    int line; /* Current line of the caret. */
    int offset; /* Character offset on the line of the caret. */

    int selStartLine = -1; /* Selection starting line. */
    int selStartOffset; /* Selection starting offset. */
    int selEndLine; /* Selection ending line. */
    int selEndOffset; /* Selection ending line. */

    int lineOffset; /* Scrolling offset for the first line. */
    int hscroll; /* Horizontal pixel offset. */

    /** The type of an undo. */
    
    enum UndoType
    {
        /** Undo marker.  When an undo command is processed, it will undo as
          * many steps as needed to get to either the next marker or the beginning
          * of the undo records.
          */
          
        Mark,
        
        SetLine, /**< Set an indexed document line to the given text. */
        SetModified, /**< Mark the document as modified. */
        InsertLine, /**< Insert a text line before the given index. */
        SetClean, /**< Mark the document as unmodified. */
        DeleteLine, /**< Delete an indexed text line. */
    }

    /* Record for undo.  This is an action to be done to reverse whatever
     * changes were made, rather than a description of the action that was
     * done.
     */

    struct UndoRecord
    {
        UndoType type; /**< Type of the undo record. */
        int index; /* Line that this record affects. */

        union
        {
            int offset; /* Offset for mark command. */
            char [] text; /* Relevant text. */
        }

        /* Perform, turning it into a redo or back again. */
        void perform (Document doc)
        {
            char [] prev;

            doc.dirty = true;
            switch (type)
            {
                case UndoType.SetLine: // Set line to text.
                    prev = doc.lines [index];
                    doc.lines [index] = text;
                    type = UndoType.SetLine;
                    text = prev;
                    doc.dirtyFrom (index);
                    if (global.view.document === doc)
                        global.view.dirtyChangeLine (index);
                    break;

                case UndoType.SetModified: // mark document as modified.
                    type = doc.modified ? UndoType.SetModified : UndoType.SetClean;
                    doc.modified = true;
                    break;

                case UndoType.InsertLine: // add text line.
                    doc.lines ~= null;
                    doc.highs ~= null;
                    for (int c = doc.lines.length - 1; c > index; c --)
                        doc.lines [c] = doc.lines [c - 1];
                    doc.lines [index] = text;
                    type = UndoType.DeleteLine;
                    doc.dirtyFrom (index);
                    if (global.view.document === doc)
                        global.view.dirtyFromLine (index);
                    break;

                case UndoType.DeleteLine: // delete text line.
                    text = doc.lines [index];
                    doc.deleteLine (index);
                    type = UndoType.InsertLine;
                    doc.dirtyFrom (index);
                    if (global.view.document === doc)
                        global.view.dirtyFromLine (index);
                    break;

                case UndoType.SetClean: // mark document as unmodified.
                    type = doc.modified ? UndoType.SetModified : UndoType.SetClean;
                    doc.modified = false;
                    break;
            }
        }
    }

    UndoRecord [] undoRecords;
    int undoPoint; /* Tracks undoRecords.length */

    this ()
    {
        setText ("");
        syntaxHighlighter ("Text File");
        index = newIndex ++;
    }

    this (char [] filename)
    {
        char [] data = (char []) read (filename);

        this.filename = filename;
        setText (data);
        guessSyntaxHighlighter ();
    }
    
    /** Guess the syntax highlighter based on the extension. */
    void guessSyntaxHighlighter ()
    {
        SyntaxHighlighter [] list = SyntaxHighlighter.list;
        SyntaxHighlighter match;
        float matchCompare;

        for (int c; c < list.length; c ++)
        {
            float compare = list [c].match (filename, lines);

            if (!compare)
                continue;
            if (match === null || compare > matchCompare)
            {
                matchCompare = compare;
                match = list [c];
            }
        }

        if (match !== null)
            syntaxHighlighter (match);
        else
            syntaxHighlighter ("Text File");
    }

    char [] name ()
    {
        static char [64] buffer;

        if (filename !== null)
            return digCommonGetBaseName (filename);
        int length = std.c.stdio.sprintf (buffer, "New File #%d", index);
        return buffer [0 .. length];
    }

    /* Add an undo record. */
    void undoAdd (UndoRecord rec)
    {
        undoRecords = undoRecords [0 .. undoPoint];
        undoRecords ~= rec;
        undoPoint ++;
    }

    /* Assign the syntax highlighter. */
    void syntaxHighlighter (SyntaxHighlighter value)
    {
        hilite = value;
        hiliteExtra = hilite.extraSize ();
        for (int c; c < highs.length; c ++)
            highs [c] = null;
        dirtyAll ();
        if (global.view.document === this)
            global.view.paint ();
    }

    /* Assign the syntax highlighter. */
    void syntaxHighlighter (char [] name)
    {
        SyntaxHighlighter [] list = SyntaxHighlighter.list;

        for (int c; c < list.length; c ++)
        {
            char [] other = list [c].name ();

            if (other == name)
            {
                syntaxHighlighter (list [c]);
                return;
            }
        }

        throw new Error ("Couldn't find syntax highlighter named '" ~ name ~ "'.");
    }

    /* Dirty all lines. */
    void dirtyAll ()
    {
        dirtyRange (0, lines.length);
    }

    /* Dirty syntax highlighting for a range of lines. */
    void dirtyRange (int start, int end)
    {
        assert (start >= 0);
        assert (end <= lines.length);
        assert (start <= end);
    
        if (start >= lines.length)
            return;
        highDirty (start, 2);
        for (int c = start + 1; c < end; c ++)
            highDirty (c, 1);
    }

    /* Dirty syntax highlighting for a single line. */
    void dirtyLine (int index)
    {
        assert (index >= 0 && index < lines.length);
        highDirty (index, 2);
    }

    /* Dirty all lines from this point. */
    void dirtyFrom (int index)
    {
        dirtyRange (index, lines.length);
    }

    /* Setup syntax highlighting for a range of lines. */
    void cleanRange (int start, int end)
    {
        assert (start >= 0);
        assert (end < lines.length);
        assert (start <= end);
        for (int c = end; c >= start; c --)
            cleanLine (c);
    }
    
    /** Return pointer to the extra highlighting info for a line. */
    void *highExtra (int index)
    {
        if (highs [index] === null)
            return null;
        return (char *) highs [index] + highs [index].length - hiliteExtra;
    }
    
    /** Return whether a highlighting line is dirty. */
    char highDirty (int index)
    {
        if (highs [index] === null)
            return 2;
        return highs [index] [highs [index].length - hiliteExtra - 1];
    }
    
    /** Assign whether a highlighting line is dirty.  0 means that it's clean, 1 means
      * that it's somewhat dirty (will clean if the previous line's extra data is changed),
      * 2 means that it's very dirty (will always clean).
      */
    void highDirty (int index, char value)
    {
        if (highs [index] === null)
            return;
        highs [index] [highs [index].length - hiliteExtra - 1] = value;
    }

    /* Perform syntax highlighting for a single line and return whether the extra data changed. */
    bit cleanLine (int index)
    {
        char [] line, high; /* Text line string and output high. */
        void *prev, old, next;
        char currentDirty;

        assert (index >= 0 && index < highs.length);

        if ((currentDirty = highDirty (index)) == 0)
            return false;

        if (index)
        {
            if (highDirty (index - 1) && !cleanLine (index - 1) && currentDirty == 1 && line.length == high.length - hiliteExtra - 1)
                return false;
            prev = highExtra (index - 1);
        }

        old = highExtra (index);
        line = lines [index];
        highs [index] = high = new char [line.length + 1 + hiliteExtra];
        next = highExtra (index);

        hilite.highlight (line, high [0 .. line.length], prev, next);
        if (hiliteExtra == 0)
            return false;
        if (old !== null && next !== null)
            return ((ubyte *) old) [0 .. hiliteExtra] != ((ubyte *) next) [0 .. hiliteExtra];
        return true;
    }

    /* Delete an inclusive range of lines. */
    void deleteRange (int start, int end)
    {
        assert (start >= 0);
        assert (end <= lines.length);
        assert (start <= end);

        for (int c = end; c < lines.length; c ++)
            lines [c - end + start] = lines [c];
        for (int c = lines.length - (end - start); c < lines.length; c ++)
            lines [c] = null;
        lines = lines [0 .. lines.length - (end - start)];

        for (int c = end; c < highs.length; c ++)
            highs [c - end + start] = highs [c];
        for (int c = highs.length - (end - start); c < highs.length; c ++)
            highs [c] = null;
        highs = highs [0 .. highs.length - (end - start)];

        /* Ensure collection. */
        if (lines.length == 0)
            lines = highs = null;

        dirty = true;
    }

    /* Delete a single line. */
    void deleteLine (int index)
    {
        deleteRange (index, index + 1);
        dirty = true;
    }

    /* Clear all text and assign new text. */
    void setText (char [] string)
    {
        deleteRange (0, lines.length);
        lines = digCommonFullSplitLines (string);
        highs.length = lines.length;
        for (int c; c < highs.length; c ++)
            highs [c] = null;
        dirty = true;
    }

    bit selectFilename ()
    {
        SyntaxHighlighter [] list = SyntaxHighlighter.list;
        char [] [] result;

        with (new FileSelector (true))
        {
            addFilter ("All Files", "*");
            for (int c; c < list.length; c ++)
            {
                char [] exts = list [c].exts ();
                char [] name = list [c].name ();

                if (exts !== null)
                    addFilter (name, exts);
            }

            result = run ();
            if (result === null)
                return false;
        }

        filename = result [0];
        return true;
    }

    /** Save to the filename. */
    void save ()
    {
        if (filename === null)
        {
            if (!selectFilename ())
                return;
            guessSyntaxHighlighter ();
            global.documentRenamed (this);
        }
        saveTo (filename);
        setModified (false);
    }

    /* Save to the given filename, doesn't change any state. */
    void saveTo (char [] filename)
    {
        char [] text;
        int length;

        for (int c; c < lines.length; c ++)
        {
            length += lines [c].length;
            if (c)
                length += endline.length;
        }
        
        text.length = length;
        length = 0;
        for (int c; c < lines.length; c ++)
        {
            text [length .. length + lines [c].length] = lines [c];
            length += lines [c].length;
            if (c < lines.length - 1)
            {
                text [length .. length + endline.length] = endline;
                length += endline.length;
            }
        }

        std.file.write (filename, (byte []) text);
    }

    override int opCmp (Object o)
    {
        Document d = cast (Document) o;

        if (cast (Document) o === null)
            return 0;

        if (filename === null)
        {
            if (d.filename === null)
                return index - d.index;
            else
                return -1;
        }
        else if (d.filename === null)
            return 1;
        else
            return std.string.cmp (digCommonGetBaseName (filename), digCommonGetBaseName (d.filename));
    }

/* undo recording: */

    /* Set modified, with undo recording if it's been changed. */
    void setModified (bit value)
    {
        if (modified == value)
            return;
        modified = value;

        UndoRecord rec;

        rec.type = modified ? UndoType.SetClean : UndoType.SetModified;
        undoAdd (rec);
    }

    /* Set a line, with undo recording. */
    void setLine (int index, char [] text)
    {
        UndoRecord rec;

        setModified (true);
       
        /* Search for a record before the next mark that also modifies this
         * line.  If one exists, the command is globbed into it.
         */
        
        for (int c = undoPoint - 1; c >= 0; c --)
        {
            UndoRecord *compare = &undoRecords [c];
            
            if (compare.type == UndoType.Mark)
                break;
            if (compare.type == UndoType.SetLine
             && compare.index == index)
                goto finish;
        }
        
        /* We didn't find a duplicate record, so continue with the insertion. */
        rec.type = UndoType.SetLine;
        rec.index = index;
        rec.text = lines [index];
        undoAdd (rec);

    finish:
        lines [index] = text;
        dirtyFrom (index);
        dirty = true;
        
        if (global.view.document === this)
            global.view.dirtyChangeLine (index);
    }

    /* Delete a line, with undo recording. */
    void delLine (int index)
    {
        UndoRecord rec;

        setModified (true);
        rec.type = UndoType.InsertLine;
        rec.index = index;
        rec.text = lines [index];
        undoAdd (rec);

        deleteLine (index);
        dirty = true;
        
        if (global.view.document === this)
            global.view.dirtyFromLine (index);
    }

    /* Insert a line at this point, with undo recording. */
    void addLine (int index, char [] text)
    {
        UndoRecord rec;

        setModified (true);
        dirtyFrom (index);
        rec.type = UndoType.DeleteLine;
        rec.index = index;
        undoAdd (rec);

        lines ~= null;
        highs ~= null;
        
        for (int c = lines.length - 1; c > index; c --)
        {
            lines [c] = lines [c - 1];
            highs [c] = highs [c - 1];
        }
        
        lines [index] = text;
        highs [index] = null;
        dirty = true;
        
        if (global.view.document === this)
            global.view.dirtyFromLine (index);
    }

    /** Insert an undo mark. */
    void mark (int line, int offset)
    {
        UndoRecord rec;
        
        rec.type = UndoType.Mark;
        rec.index = line;
        rec.offset = offset;
        undoAdd (rec);
    }

    /** Undo the most recent action. */
    void undo (inout int line, inout int offset)
    {
        undoClean ();
        if (undoPoint == 0)
            return;
        while (-- undoPoint >= 0)
        {
            UndoRecord *rec = &undoRecords [undoPoint];

            if (rec.type == UndoType.Mark)
            {
                line = rec.index;
                offset = rec.offset;
                break;
            }
            rec.perform (this);
        }

        if (undoPoint < 0)
            undoPoint = 0;
    }

    /** Redo the most recent action. */
    void redo (inout int line, inout int offset)
    {
        undoClean ();
        if (undoPoint == undoRecords.length)
            return;
        while (++ undoPoint < undoRecords.length)
        {
            UndoRecord *rec = &undoRecords [undoPoint];

            if (rec.type == UndoType.Mark)
            {
                line = rec.index;
                offset = rec.offset;
                break;
            }
            rec.perform (this);
        }
    }

    /** Remove an empty mark from the end of the undo stack. */
    void undoClean ()
    {
        while (undoRecords.length && undoRecords [undoRecords.length - 1].type == UndoType.Mark)
        {
            if (undoPoint == undoRecords.length)
                undoPoint --;
            undoRecords = undoRecords [0 .. undoRecords.length - 1];
        }
    }

    /* Determine whether the selection box is valid. */
    bit selExists ()
    {
        return selStartLine != -1;
    }

    /* Clear the selection. */
    void selClear ()
    {
        selStartLine = -1;
    }

    /* Invert the selection if necessary. */
    void selCheck ()
    {
        if (selStartLine < selEndLine)
            return;
        if (selStartLine == selEndLine && selStartOffset <= selEndOffset)
            return;

        int swap = selStartLine;
        selStartLine = selEndLine;
        selEndLine = swap;

        swap = selStartOffset;
        selStartOffset = selEndOffset;
        selEndOffset = swap;
    }

    /* Move the selection along with the cursor or start a new selection. */
    void selFollowCursor (int aline, int aoffset, int bline, int boffset)
    {
        if (selExists () && aline == selStartLine && aoffset == selStartOffset)
        {
            selStartLine = bline;
            selStartOffset = boffset;
        }
        else if (selExists () && aline == selEndLine && aoffset == selEndOffset)
        {
            selEndLine = bline;
            selEndOffset = boffset;
        }
        else
        {
            selStartLine = aline;
            selStartOffset = aoffset;
            selEndLine = bline;
            selEndOffset = boffset;
        }

        selCheck ();
    }

    /* Return the tab parameters for this document. */
    TabParams tabParams ()
    {
        return hilite.tabParams ();
    }
}
